/****************************************************************************** 
 ** HID Keyboard Demo								Create Date	2015/12/03
 **				for PIC18F14K50						Last Update	2016/03/12
 ** file name :  keyboard.c									Saka Softwares
 ** language  :  XC8 (v1.35)
 ******************************************************************************
 ** HID Keyboard のデモプログラムです。
 ** Microchip のデモプログラムと同じような動作をします。
 ** スイッチを押すたびに'a' 'b' 'c' と文字を出します。
 ** スイッチを押し続けるとキーリピートして同じ文字が出ます。(Microchipのデモでは
 ** 出ません。)
 ** 簡易のデモなのでスイッチのチャタリング防止等は組み込んでいません。
 ******************************************************************************/
/** Include Files *************************************************************/
#include <xc.h>
#include <stdint.h>
#include <stdbool.h>

#include "usbhiddev.h"

/** Version info. *************************************************************/
//注. 下記 VersionWord は、XC8 (v1.35)で旨く書き込めない場合があります。
const uint16_t VersionWord @ 0x0816 = 0x0120;	// BCD表現のバージョン No.
/******************************************************************************
 **	1.2		2016/03/12	usbhiddevの不具合修正
 **	1.0		2015/11/19	公開バージョン
 ******************************************************************************/

/** Configuration *************************************************************/
// HIDブートローダーと同一コンフィグレーションにしてください。
// pic18f14k50_xc8_bootloader.hexに対応
#pragma config  CPUDIV	= NOCLKDIV	// CPU システムクロック選択ビット
#pragma config  USBDIV	= OFF		// USB クロック選択ビット
#pragma config  FOSC	= HS		// オシレータ選択ビット
#pragma config  PLLEN	= ON		// 4 X PLL イネーブルビット
#pragma config  PCLKEN	= OFF		// プライマリ クロック イネーブルビット
#pragma config  FCMEN	= OFF		// フェイルセーフ クロック モニタ イネーブルビット
#pragma config  IESO	= OFF		// 内部/ 外部オシレータ切り換えビット
#pragma config  PWRTEN	= OFF		// パワーアップ タイマ イネーブルビット (OFFで有効)
#pragma config  BOREN	= ON		// ブラウンアウト リセット イネーブルビット
#pragma config  BORV	= 30		// ブラウンアウト リセット電圧ビット
#pragma config  WDTEN	= OFF		// ウォッチドッグ タイマ イネーブルビット
#pragma config  WDTPS	= 32768		// ウォッチドッグ タイマ ポストスケーラ選択ビット
#pragma config  HFOFST	= OFF		// HFINTOSC 高速起動ビット
#pragma config  MCLRE	= OFF		// MCLR ピン イネーブルビット
#pragma config  STVREN	= ON		// スタックフル/ アンダーフロー リセット イネーブルビット
#pragma config  LVP		= OFF		// 単電源ICSP イネーブルビット
#pragma config  BBSIZ	= ON		// ブートブロック サイズ選択ビット(2kW boot block size)
#pragma config  XINST	= OFF		// 拡張命令セットイネーブルビット (ENHCPU)
#pragma config  CP0		= OFF		// コード保護ビット
#pragma config  CP1		= OFF		// コード保護ビット
#pragma config  CPB		= OFF		// ブートブロック コード保護ビット
#pragma config  CPD		= OFF		// EEPROM コード保護ビット
#pragma config  WRT0	= OFF		// 書き込み保護ビット
#pragma config  WRT1	= OFF		// 書き込み保護ビット
#pragma config  WRTC	= OFF		// コンフィグレーション レジスタ書き込み保護ビット
#pragma config  WRTB	= OFF		// ブートブロック書き込み保護ビット
#pragma config  WRTD	= OFF		// EEPROM 書き込み保護ビット
#pragma config  EBTR0	= OFF		// テーブル読み出し保護ビット
#pragma config  EBTR1	= OFF		// テーブル読み出し保護ビット
#pragma config  EBTRB	= OFF		// ブートブロック テーブル読み出し保護ビット

/** Hardware Settings *********************************************************/
#define	INPUT	1
#define	OUTPUT	0
// Indicator
#define NumLockIndicator	LATCbits.LATC0
#define NumLock_Tris		TRISCbits.TRISC0
#define CapsLockIndicator	LATCbits.LATC1
#define CapsLock_Tris		TRISCbits.TRISC1
#define ScrollLockIndicator	LATCbits.LATC2
#define ScrollLock_Tris		TRISCbits.TRISC2
//#define KanaIndicator		LATxbits.LATxx
//#define Kana_Tris			TRISxbits.TRISxx

// SWITCH
#define sw2			PORTAbits.RA3

/** Variable ******************************************************************/
// キーボード インプット レポート
typedef union _KEY_IN_REPORT {
    uint8_t buf[8];		//バッファ
    struct {
		struct {
			unsigned leftControl	:1;	//左コントロールキー
			unsigned leftShift		:1;	//左シフトキー
			unsigned leftAlt		:1;	//左Altキー
			unsigned leftGUI		:1;	//左GUI(Win/Command)キー
			unsigned rightControl	:1;	//右コントロールキー
			unsigned rightShift		:1;	//右シフトキー
			unsigned rightAlt		:1;	//右Altキー
			unsigned rightGUI		:1;	//右GUI(Win/Command)キー
		}bits;
		unsigned :8;
		uint8_t key[6];					//キーコード
	};
} KEY_IN_REPORT;
KEY_IN_REPORT keyInRepote;				// 送信バッファ兼用

// キーボード アウトプット レポート
typedef union _KEY_OUT_REPORT {
	uint8_t buf[8];						//バッファ
	struct {
		unsigned numlock	:1;			//numlock フラグ
		unsigned capslock	:1;			//capslock フラグ
		unsigned compose	:1;			//ScrollLock フラグ
		unsigned kana		:1;			//カナ フラグ
		unsigned :4;
	}bits;								//フラグbits
} KEY_OUT_REPORT;
KEY_OUT_REPORT keyOutRepote;			// 受信バッファ兼用

uint32_t NextSOFcount;					// 次のSOFカウント
uint8_t code;							// キーコード
bool bsw2;								// スイッチ2フラグ

/** Fixed number **************************************************************/

/** Internal Function *********************************************************/
void main(void);
void interrupt low_priority IsrLow(void);
void KeybordInit(void);
void KeyboardIndicator(KEY_OUT_REPORT rpt);
void ProcessIO(void);
void MainUsbStatus(void);

/******************************************************************************
 ** メイン処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ******************************************************************************/
void main(void)
{
	ClrWdt();
	INTCON  = 0x00;					// 割り込み禁止
	STKPTR  = 0x00;					// スタックポインタ設定
	WDTCON  = 0x01;					// ウォッチドックタイマ起動
	RCONbits.IPEN	= 1;			// 優先度レベルの設定を有効にす
	IPR2bits.USBIP	= 0;			// USB 割り込み優先度:Low

	KeybordInit();					// キーボード初期化
	
	USBDeviceInit();				// 割り込み対応USBの初期化
	INTCONbits.GIEL = 1;			// 低割り込み許可
	INTCONbits.GIEH = 1;			// 高割り込み許可
	while (1) {
		ClrWdt();
		USBDeviceTasks();			// USBのバス接続切断処理を行います。
		MainUsbStatus();			// USBフラグを見て処理を行います。
		ProcessIO();				// 通信処理
    }    
}    

/******************************************************************************
 ** USB割り込み処理部処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** USBのプラグ&プレイ関連の処理を行っています。必ず必要です。 
 ******************************************************************************/
void interrupt low_priority IsrLow(void)
{
	// USB割り込み処理
	USBInterruptTasks();
}

/******************************************************************************
 ** キーボード処理初期化部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 **  キーボード処理の初期化を行います。
 ******************************************************************************/
void KeybordInit(void)
{
	// 表示器の初期化
#ifdef NumLockIndicator
	NumLock_Tris	 = OUTPUT;
	NumLockIndicator = 0;
#endif
#ifdef CapsLockIndicator
	CapsLock_Tris	  = OUTPUT;
	CapsLockIndicator = 0;
#endif
#ifdef ScrollLockIndicator
	ScrollLock_Tris		= OUTPUT;
	ScrollLockIndicator = 0;
#endif
#ifdef KanaIndicator
	Kana_Tris	  = OUTPUT;
	KanaIndicator = 0;
#endif
	// レポートの初期化
	uint8_t i = 8;
	while (i--) {
		keyInRepote.buf[i]	= 0;
		keyOutRepote.buf[i] = 0;
	}
	// 通信開始を500mS先を設定
	NextSOFcount = USBGetSOFCounter() + 500;
	code = 4;				// 'a' キーのコード
	bsw2 = false;			// スイッチは押されていない
}

/******************************************************************************
 ** キーボード表示器処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 **  キーボード処理の初期化を行います。
 ******************************************************************************/
void KeyboardIndicator(KEY_OUT_REPORT rpt)
{
#ifdef NumLockIndicator
	if (rpt.bits.numlock)	NumLockIndicator = 1;
	else					NumLockIndicator = 0;
#endif
#ifdef CapsLockIndicator
	if (rpt.bits.capslock)	CapsLockIndicator = 1;
	else					CapsLockIndicator = 0;
#endif
#ifdef ScrollLockIndicator
	if (rpt.bits.compose)	ScrollLockIndicator = 1;
	else					ScrollLockIndicator = 0;
#endif
#ifdef KanaIndicator
	if (rpt.bits.kana)		KanaIndicator = 1;
	else					KanaIndicator = 0;
#endif
}

/******************************************************************************
 ** プロセスＩＯ処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** USB関連の処理を一括で行っています。
 ** HID受信した場合フラグを立てて各処理を行い、HID送信が可能になるまで待ち
 ** 結果を返す仕様にしています。
 ** 1回の受信または送信は64byteです。HIDのリポートディスクリプタで設定しています。
 ** 送信処理(HidTxRamReport)は、前の送信が終わるまで待ちます。
 ** 複数送る場合ウォッチドックのタイムアウトに気を付けてください。
 ******************************************************************************/
void ProcessIO(void)
{
	// SOFのカウント数を取得
	uint32_t SOFCount = USBGetSOFCounter();
	bool send = false;
	// キー操作
	if (!sw2 && !bsw2) {
		bsw2 = true;
		keyInRepote.key[0] = code++;
		if (code == 40)	code = 4;
		send = true;
	}
	else if (sw2 && bsw2) {
		bsw2 = false;
		keyInRepote.key[0] = 0;
		send = true;
	}
	// 送信
	if (send || NextSOFcount == SOFCount) {
		NextSOFcount = SOFCount + USBGetIdleTime();
		HidTxRamReport(keyInRepote.buf, 8);
	}
	// 受信
	if (HidRxReport(keyOutRepote.buf, 8) < RCV_RECEING) {
		KeyboardIndicator(keyOutRepote);
	}
}

/******************************************************************************
 ** USBステータスによる各処理部
 ** 引  数	: 無し
 ** 戻り値	: 無し
 ** LED表示処理
 ** USBに未接続時は消灯 USB接続処理中は点灯 USBの通信中は点滅します。
 ** サスペンド処理
 ** サスペンド時にスリープします。
 ** その前後にユーザー処理を入れます。
 ******************************************************************************/
void MainUsbStatus(void)
{
	//** サスペンド処理 ********************************************************
	if (USBGetDeviceState() == CONFIGURED_STATE && USBIsSuspended() ) {
		// メインのサスペンド移行処理部
		KEY_OUT_REPORT rpt;
		rpt.buf[0] = 0;
		KeyboardIndicator(rpt);					// 表示器を消灯
		// サスペンド中何もせずスリープする
		while (!USBIsActive() && !RCONbits.TO) {
			Sleep();
		}
		// メインのウェイクアップ処理部
		KeyboardIndicator( keyOutRepote );		// 表示器を復帰
	}
}

/** EOF main.c ****************************************************************/
